<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../config/db.php';

// ==========================================================
// Lee parámetros: desde JSON (POST) o query string.
// Campos esperados:
//   - label   (opcional)
//   - type    ("billete" | "moneda")
//   - value   (int)
//   - family  (string, opcional)
//   - year    (int, opcional)
//   - variant (string, opcional)
// ==========================================================
$label  = '';
$type   = '';
$variant= '';
$family = '';
$value  = 0;
$year   = 0;

$ct = $_SERVER['CONTENT_TYPE'] ?? '';

if (stripos($ct, 'application/json') !== false) {
    $j       = json_decode(file_get_contents('php://input'), true) ?: [];
    $label   = trim((string)($j['label']   ?? ''));
    $type    = trim((string)($j['type']    ?? ''));
    $value   = (int)($j['value']          ?? 0);
    $year    = (int)($j['year']           ?? 0);
    $variant = trim((string)($j['variant'] ?? ''));
    $family  = trim((string)($j['family']  ?? ''));
} else {
    $label   = trim((string)($_REQUEST['label']   ?? ''));
    $type    = trim((string)($_REQUEST['type']    ?? ''));
    $value   = (int)($_REQUEST['value']          ?? 0);
    $year    = (int)($_REQUEST['year']           ?? 0);
    $variant = trim((string)($_REQUEST['variant'] ?? ''));
    $family  = trim((string)($_REQUEST['family']  ?? ''));
}

// Pequeño helper por si solo llega label y no type/value
function parseLabel(string $label): array
{
    $label = trim($label);
    if ($label === '') {
        return [null, null];
    }

    // Ejemplos aceptados:
    //  "billete 1000", "moneda 50", "1000 billete", "moneda_200"
    if (preg_match('~\b(billete|moneda)\b[^0-9]*([0-9]+)~i', $label, $m)) {
        return [ strtolower($m[1]), (int)$m[2] ];
    }

    if (preg_match('~([0-9]+)~', $label, $m)) {
        // Tenemos valor pero no tipo.
        return [ null, (int)$m[1] ];
    }

    return [null, null];
}

// Si tenemos label pero no type/value, intentamos inferirlos.
if ($label && (!$type || !$value)) {
    [$t, $v] = parseLabel($label);
    if ($t && !$type) {
        $type = $t;
    }
    if ($v && !$value) {
        $value = $v;
    }
}

if (!$type || !$value) {
    echo json_encode(['ok' => false, 'msg' => 'invalid params (type/value)']);
    exit;
}

// Normalizamos un poco
$type    = strtolower($type);
$variant = strtolower($variant);
$family  = strtoupper($family);

// ==========================================================
// Construimos el SQL dinámicamente:
// - Siempre filtramos por type y value
// - Opcionalmente family
// - Opcionalmente un rango de años (year_between)
// - Opcionalmente variant
// ==========================================================
$sql    = "SELECT * FROM denominations WHERE type = ? AND value = ?";
$params = [$type, $value];

if ($family !== '') {
    $sql     .= " AND family = ?";
    $params[] = $family;
}

if ($year > 0) {
    // año dentro del rango [year_from, year_to] si existen.
    $sql     .= " AND ( (year_from IS NULL OR year_from <= ?) AND (year_to IS NULL OR year_to >= ?) )";
    $params[] = $year;
    $params[] = $year;
}

if ($variant !== '') {
    $sql     .= " AND LOWER(variant) = ?";
    $params[] = $variant;
}

// Si hay varias, preferimos las más específicas / recientes
$sql .= " ORDER BY COALESCE(year_from, 0) DESC, id DESC LIMIT 1";

try {
    $st  = $pdo->prepare($sql);
    $st->execute($params);
    $row = $st->fetch();
} catch (Throwable $e) {
    echo json_encode([
        'ok'    => false,
        'msg'   => 'DB error',
        'detail'=> $e->getMessage(),
    ]);
    exit;
}

if (!$row) {
    echo json_encode(['ok' => false, 'msg' => 'not found']);
    exit;
}

// Devolvemos la URL de referencia (por ejemplo, de numismaticol.com)
echo json_encode([
    'ok'      => true,
    'url'     => $row['ref_url'],
    'matched' => [
        'type'    => $row['type'],
        'value'   => (int)$row['value'],
        'family'  => $row['family'],
        'year_from' => $row['year_from'],
        'year_to'   => $row['year_to'],
        'variant' => $row['variant'],
    ],
]);
