<?php
require_once __DIR__.'/env.php';

/**
 * Envía una o varias imágenes al modelo de visión de OpenAI y devuelve
 * un array con:
 *  - ok: true/false
 *  - data: array decodificado desde el JSON que devuelve el modelo
 *  - raw:  texto crudo devuelto por el modelo (string)
 *  - msg/detail en caso de error
 *
 * @param string[] $images_dataurls  Arreglo de data URLs (data:image/xxx;base64,...)
 * @param string   $instruction      Texto con las instrucciones para el modelo
 */
function openai_chat_vision_extract(array $images_dataurls, string $instruction)
{
    $apiKey = OPENAI_API_KEY;
    if (!$apiKey || $apiKey === 'REEMPLAZAR_OPENAI_API_KEY') {
        http_response_code(500);
        echo json_encode(['ok' => false, 'msg' => 'Falta configurar OPENAI_API_KEY']);
        exit;
    }

    if (empty($images_dataurls)) {
        return ['ok' => false, 'msg' => 'No se recibió ninguna imagen'];
    }

    $model = OPENAI_VISION_MODEL ?: 'gpt-4o-mini';

    // Contenido del mensaje del usuario: instrucciones + imágenes
    $content = [
        [
            'type' => 'text',
            'text' => $instruction,
        ],
    ];

    foreach ($images_dataurls as $d) {
        $content[] = [
            'type'      => 'image_url',
            'image_url' => [ 'url' => $d ],
        ];
    }

    $payload = [
        'model'    => $model,
        'messages' => [
            [
                'role'    => 'system',
                'content' => 'Eres un experto en numismática colombiana. ' .
                             'Siempre respondes únicamente JSON válido, sin explicaciones adicionales.',
            ],
            [
                'role'    => 'user',
                'content' => $content,
            ],
        ],
        'temperature' => 0.1,
        'max_tokens'  => 400,
    ];

    $ch = curl_init('https://api.openai.com/v1/chat/completions');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $apiKey,
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload),
        // Si tu servidor tiene problemas de certificados, puedes descomentar
        // estas dos líneas BAJO TU RESPONSABILIDAD (no recomendado en producción):
        // CURLOPT_SSL_VERIFYHOST => 0,
        // CURLOPT_SSL_VERIFYPEER => 0,
    ]);

    $res  = curl_exec($ch);
    $err  = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($res === false) {
        return ['ok' => false, 'msg' => 'Error cURL', 'detail' => $err];
    }

    $j = json_decode($res, true);
    if (!is_array($j)) {
        return ['ok' => false, 'msg' => 'Respuesta no JSON de OpenAI', 'raw' => $res];
    }

    if ($http >= 400) {
        return [
            'ok'    => false,
            'msg'   => 'OpenAI HTTP ' . $http,
            'detail'=> $j,
        ];
    }

    $contentText = $j['choices'][0]['message']['content'] ?? '';
    if (!is_string($contentText)) {
        return ['ok' => false, 'msg' => 'OpenAI no devolvió contenido de texto', 'raw' => $contentText];
    }

    // Por si el modelo envía ```json ... ```
    if (preg_match('~```json\s*(.*?)\s*```~si', $contentText, $m)) {
        $contentText = $m[1];
    }

    $data = json_decode($contentText, true);
    if ($data === null) {
        return [
            'ok'   => false,
            'msg'  => 'OpenAI no devolvió JSON parseable',
            'raw'  => $contentText,
            'full' => $j,
        ];
    }

    return [
        'ok'  => true,
        'data'=> $data,
        'raw' => $contentText,
    ];
}
