const video = document.getElementById('video');
const canvas = document.getElementById('frame');
const ctx = canvas.getContext('2d');
const preview = document.getElementById('preview');

const btnStart = document.getElementById('btn-start');
const btnAI = document.getElementById('btn-ai');
const btnUpload = document.getElementById('btn-upload');
const fileInput = document.getElementById('file');

const statusBadge = document.getElementById('status-badge');

const outType = document.getElementById('out-type');
const outValue = document.getElementById('out-value');
const outFamily = document.getElementById('out-family');
const outYear = document.getElementById('out-year');
const outVariant = document.getElementById('out-variant');
const outConf = document.getElementById('out-conf');
const outParsed = document.getElementById('out-parsed');
const outCands = document.getElementById('out-cands');
const outUrl = document.getElementById('out-url');

const selType = document.getElementById('sel-type');
const selValue = document.getElementById('sel-value');
const inpFamily = document.getElementById('inp-family');
const inpYear = document.getElementById('inp-year');
const inpVariant = document.getElementById('inp-variant');
const btnManual = document.getElementById('btn-manual');
const btnOpenManual = document.getElementById('btn-open-manual');
const btnFillAI = document.getElementById('btn-fill-ai');

let stream = null;
let lastAI = null;
let lastGuideUrl = null;

function setStatus(text) {
  statusBadge.textContent = text;
}

// ======================================================
// Cámara
// ======================================================
async function startCamera() {
  try {
    stream = await navigator.mediaDevices.getUserMedia({ video: { facingMode: 'environment' }, audio: false });
    video.srcObject = stream;
    await video.play();
    setStatus('Cámara encendida');
  } catch (e) {
    console.error(e);
    setStatus('Error cámara: ' + e.message);
  }
}

function grabDataUrlFromVideo() {
  const w = video.videoWidth;
  const h = video.videoHeight;
  if (!w || !h) {
    throw new Error('Video no listo todavía');
  }
  canvas.width = w;
  canvas.height = h;
  ctx.drawImage(video, 0, 0, w, h);
  const dataUrl = canvas.toDataURL('image/jpeg', 0.9);
  preview.src = dataUrl;
  preview.style.display = 'block';
  return dataUrl;
}

// ======================================================
// Llamadas a la API
// ======================================================
async function aiIdentifyByDataUrl(dataUrl) {
  setStatus('Analizando con IA...');
  const r = await fetch('../api/ai_identify.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ image_data: dataUrl }),
  });
  return r.json();
}

async function aiIdentifyByFile(file) {
  setStatus('Analizando con IA...');
  const form = new FormData();
  form.append('file', file);
  const r = await fetch('../api/ai_identify.php', {
    method: 'POST',
    body: form,
  });
  return r.json();
}

async function resolveGuide(payload) {
  setStatus('Buscando guía...');
  const r = await fetch('../api/resolve.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(payload),
  });
  const j = await r.json();
  if (!j.ok) return null;
  return j.url;
}

// ======================================================
// UI helpers
// ======================================================
function showAI(ai) {
  lastAI = ai;

  const type = ai.type || '';
  const value = ai.value != null ? ai.value : '';
  const family = ai.family || '';
  const year = ai.year || '';
  const variant = ai.variant || '';
  const confidence = ai.confidence != null ? ai.confidence : null;

  outType.textContent = type || '—';
  outValue.textContent = value || '—';
  outFamily.textContent = family || '—';
  outYear.textContent = year || '—';
  outVariant.textContent = variant || '—';

  if (confidence == null) {
    outConf.textContent = '—';
  } else {
    const pct = Math.round(confidence * 1000) / 10;
    outConf.textContent = confidence.toFixed(3) + ' (' + pct + '%)';
  }

  const summaryParts = [];
  if (type) summaryParts.push(type);
  if (value) summaryParts.push(value + ' pesos');
  if (family) summaryParts.push('familia ' + family);
  if (year) summaryParts.push('año ' + year);
  if (variant) summaryParts.push('variante ' + variant);
  outParsed.textContent = summaryParts.length ? summaryParts.join(' — ') : 'Sin datos';

  if (Array.isArray(ai.candidates) && ai.candidates.length) {
    outCands.innerHTML = ai.candidates
      .slice(0, 3)
      .map((c, idx) => {
        const pieces = [];
        if (c.type) pieces.push(c.type);
        if (c.value != null) pieces.push(c.value + ' pesos');
        if (c.family) pieces.push('fam ' + c.family);
        if (c.year) pieces.push('año ' + c.year);
        if (c.variant) pieces.push('var ' + c.variant);
        if (c.confidence != null) {
          const p = Math.round(c.confidence * 1000) / 10;
          pieces.push('conf ' + p + '%');
        }
        return (idx + 1) + '. ' + pieces.join(' — ');
      })
      .join('\n');
  } else {
    outCands.textContent = 'Sin candidatos alternativos';
  }

  // Rellenamos automáticamente la sección manual (sin forzar al usuario)
  selType.value = type || '';
  selValue.value = value || '';
  inpFamily.value = family || '';
  inpYear.value = year || '';
  inpVariant.value = variant || '';

  setStatus('IA lista, revisa los datos');
}

function clearGuideUrl() {
  lastGuideUrl = null;
  btnOpenManual.disabled = true;
  outUrl.textContent = '—';
}

// ======================================================
// Eventos
// ======================================================
btnStart?.addEventListener('click', () => {
  startCamera();
});

btnAI?.addEventListener('click', async () => {
  try {
    const dataUrl = grabDataUrlFromVideo();
    const j = await aiIdentifyByDataUrl(dataUrl);
    if (!j.ok) {
      setStatus('IA: ' + (j.msg || 'error'));
      return;
    }
    showAI(j.ai);
    const url = await resolveGuide({
      type: j.ai.type || '',
      value: j.ai.value || null,
      family: j.ai.family || '',
      year: j.ai.year || null,
      variant: j.ai.variant || '',
    });
    if (url) {
      lastGuideUrl = url;
      btnOpenManual.disabled = false;
      outUrl.innerHTML = '<a class="link" href="' + url + '" target="_blank" rel="noreferrer">' + url + '</a>';
      setStatus('Guía encontrada');
    } else {
      clearGuideUrl();
      outUrl.textContent = 'No encontrada (ajusta familia/año/variante)';
      setStatus('Revisa datos antes de buscar de nuevo');
    }
  } catch (e) {
    console.error(e);
    setStatus('Error al capturar imagen: ' + e.message);
  }
});

btnUpload?.addEventListener('click', (e) => {
  e.preventDefault();
  fileInput?.click();
});

fileInput?.addEventListener('change', async (e) => {
  const f = e.target.files[0];
  if (!f) return;
  const reader = new FileReader();
  reader.onload = async function(ev) {
    const dataUrl = ev.target.result;
    preview.src = dataUrl;
    preview.style.display = 'block';
    const j = await aiIdentifyByDataUrl(dataUrl);
    if (!j.ok) {
      setStatus('IA: ' + (j.msg || 'error'));
      return;
    }
    showAI(j.ai);
    const url = await resolveGuide({
      type: j.ai.type || '',
      value: j.ai.value || null,
      family: j.ai.family || '',
      year: j.ai.year || null,
      variant: j.ai.variant || '',
    });
    if (url) {
      lastGuideUrl = url;
      btnOpenManual.disabled = false;
      outUrl.innerHTML = '<a class="link" href="' + url + '" target="_blank" rel="noreferrer">' + url + '</a>';
      setStatus('Guía encontrada');
    } else {
      clearGuideUrl();
      outUrl.textContent = 'No encontrada (ajusta familia/año/variante)';
      setStatus('Revisa datos antes de buscar de nuevo');
    }
  };
  reader.readAsDataURL(f);
});

btnManual?.addEventListener('click', async () => {
  const type = selType.value;
  const value = selValue.value ? parseInt(selValue.value, 10) : null;
  const family = (inpFamily.value || '').trim().toUpperCase();
  const year = inpYear.value ? parseInt(inpYear.value, 10) : null;
  const variant = (inpVariant.value || '').trim().toLowerCase();

  clearGuideUrl();

  const url = await resolveGuide({ type, value, family, year, variant });
  if (url) {
    lastGuideUrl = url;
    btnOpenManual.disabled = false;
    outUrl.innerHTML = '<a class="link" href="' + url + '" target="_blank" rel="noreferrer">' + url + '</a>';
    setStatus('Guía encontrada');
  } else {
    outUrl.textContent = 'No encontrada';
    setStatus('Sin coincidencias, ajusta los filtros');
  }
});

btnOpenManual?.addEventListener('click', () => {
  if (!lastGuideUrl) return;
  window.open(lastGuideUrl, '_blank', 'noopener');
});

btnFillAI?.addEventListener('click', () => {
  if (!lastAI) return;
  selType.value = lastAI.type || '';
  selValue.value = lastAI.value != null ? lastAI.value : '';
  inpFamily.value = lastAI.family || '';
  inpYear.value = lastAI.year != null ? lastAI.year : '';
  inpVariant.value = lastAI.variant || '';
  setStatus('Campos rellenados con datos de la IA');
});

// Estado inicial
setStatus('Listo');
